/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	$Id: pgpRPCMessagePack.c,v 1.17 2001/03/24 01:10:36 jason Exp $
____________________________________________________________________________*/

#include "pgpPubTypes.h"
#include "pgpMem.h"
#include "pgpFileSpec.h"
#include "pgpRPCMsg.h"

static char sRPCInitBuf[1024];

PGPBoolean
PGPIsInitRPCBuf(char *ptr)
{
	if (ptr == sRPCInitBuf)
		return TRUE;
	else
		return FALSE;
}

void
PGPGetRPCInitBuf(void **base, int *len)
{
	*base = sRPCInitBuf;
	*len = sizeof(sRPCInitBuf);
}

void
pack_init(PGPPackMsg *pkt, PGPInt32 type, PGPInt32 connection_ref)
{
	PGPMemoryMgrRef memref;

	memref = PGPGetDefaultMemoryMgr();
	if (memref == kInvalidPGPMemoryMgrRef)
		return;

	pkt->length = 128;
	pkt->base =  PGPNewData(memref, pkt->length, 0);
	pkt->ptr = 0;
	pack_int32(pkt, type);
	pack_int32(pkt, connection_ref);
	pack_int32(pkt, 0);			/* Length */
}

/* Used when we will be storing sensitive data in the pkt, like passphrase */
void
pack_init_secure(PGPPackMsg *pkt, PGPInt32 type, PGPInt32 connection_ref)
{
	PGPMemoryMgrRef memref;

	memref = PGPGetDefaultMemoryMgr();
	if (memref == kInvalidPGPMemoryMgrRef)
		return;

	pkt->length = 128;
	pkt->base =  PGPNewSecureData(memref, pkt->length, 0);
	pkt->ptr = 0;
	pack_int32(pkt, type);
	pack_int32(pkt, connection_ref);
	pack_int32(pkt, 0);			/* Length */
}

static void
pgpRPCPackExpand(PGPPackMsg *pkt, PGPSize inc)
{
	PGPMemoryMgrRef memref;
	int mem_needed = pkt->ptr + inc;

	if (mem_needed > pkt->length) {
		memref = PGPGetDefaultMemoryMgr();
		if (memref == kInvalidPGPMemoryMgrRef)
			return;

		if (pkt->base == (PGPByte *) sRPCInitBuf) {
			/*
			 * For the 1st expansion, don't FreeData on the existing
			 * buffer because its a static buffer.
			 */
			pkt->length = (mem_needed + 8192) / 8192 * 8192;
			pkt->base = PGPNewData(memref, pkt->length, 0);
			pgpCopyMemory(sRPCInitBuf, pkt->base, sizeof(sRPCInitBuf));
		}
		else {
			/* When we need more memory, allocate at least 8K more
			 * than we need.  Round amount to next 8K boundary.
			 */
			pkt->length = (mem_needed + 8192) / 8192 * 8192;
			PGPReallocData(memref, (void **)&pkt->base, pkt->length, 0);
		}
	}
}

void
pack_opaque(PGPPackMsg *pkt, const void *base, PGPSize len)
{
	int padded_len = (len+3) / 4 * 4;	/* 4-byte Align Data */
	if (base == NULL) len = 0;
	pgpRPCPackExpand(pkt, sizeof(PGPSize) + padded_len);
	pack_int32(pkt, len);
	if (len > 0) {
		pgpCopyMemory(base, &pkt->base[pkt->ptr], len);
		pkt->ptr += padded_len;
	}
}

void
unpack_opaque(PGPPackMsg *pkt, void **buf, PGPSize *len)
{
	int padded_len;
	/*
	 * We just set the passed-in buf ptr to the
	 * location in the source packet.
	 */
	*len = unpack_int32(pkt);
	if( *len == 0 )
		*buf = NULL;
	else
		*buf = (void *)&pkt->base[pkt->ptr];

	padded_len = (*len+3) / 4 * 4;		/* 4-byte Align Data */
	pkt->ptr += padded_len;
}

void
unpack_opaque_alloc(PGPPackMsg *pkt, void **buf, PGPSize *len)
{
	PGPMemoryMgrRef memref;

	memref = PGPGetDefaultMemoryMgr();
	if (memref == kInvalidPGPMemoryMgrRef)
		return;

	*len = unpack_int32(pkt);
	if (*len == 0)
		*buf = NULL;
	else {
		int padded_len;
		*buf = PGPNewData(memref, *len, 0);
		pgpCopyMemory(&pkt->base[pkt->ptr], *buf, *len);
		padded_len = (*len+3) / 4 * 4;		/* 4-byte Align Data */
		pkt->ptr += padded_len;
	}
}

void
unpack_opaque_secure(PGPPackMsg *pkt, void **buf, PGPSize *len)
{
	PGPMemoryMgrRef memref;

	memref = PGPGetDefaultMemoryMgr();
	if (memref == kInvalidPGPMemoryMgrRef)
		return;

	*len = unpack_int32(pkt);
	if (*len == 0)
		*buf = NULL;
	else {
		int padded_len;
		*buf = PGPNewSecureData(memref, *len, 0);
		if (!*buf)
			*buf = PGPNewData(memref, *len, 0);
		pgpCopyMemory(&pkt->base[pkt->ptr], *buf, *len);
		padded_len = (*len+3) / 4 * 4;		/* 4-byte Align Data */
		pkt->ptr += padded_len;
	}
}

void
pack_int32(PGPPackMsg *pkt, PGPInt32 num)
{
	pgpRPCPackExpand(pkt, sizeof(PGPInt32));
	*(PGPInt32 *)(&pkt->base[pkt->ptr]) = num;
	pkt->ptr += sizeof(PGPInt32);
}

PGPInt32
unpack_int32(PGPPackMsg *pkt)
{
	PGPInt32 num = *(PGPInt32 *)(&pkt->base[pkt->ptr]);
	pkt->ptr += sizeof(PGPInt32);
	return num;
}

void
pack_uint32(PGPPackMsg *pkt, PGPUInt32 num)
{
	pgpRPCPackExpand(pkt, sizeof(PGPUInt32));
	*(PGPUInt32 *)(&pkt->base[pkt->ptr]) = num;
	pkt->ptr += sizeof(PGPUInt32);
}

PGPUInt32
unpack_uint32(PGPPackMsg *pkt)
{
	PGPUInt32 num = *(PGPUInt32 *)(&pkt->base[pkt->ptr]);
	pkt->ptr += sizeof(PGPUInt32);
	return num;
}

void
pack_filespec(PGPPackMsg *pkt, PFLFileSpecRef file)
{
	PGPByte *base;
	PGPSize len;

	PFLExportFileSpec(file, &base, &len);
	pgpRPCPackExpand(pkt, len);
	pack_opaque(pkt, (PGPByte *) base, len);
	PGPFreeData(base);
}

PFLFileSpecRef
unpack_filespec(PGPPackMsg *pkt, PGPMemoryMgrRef mem_mgr)
{
	PGPByte *base;
	PGPSize len;
	PFLFileSpecRef fsr;

	unpack_opaque(pkt, (void **)&base, &len);
	PFLImportFileSpec(mem_mgr, base, len, &fsr);

	return fsr;
}

void
pack_bool(PGPPackMsg *pkt, PGPBoolean num)
{
	pgpRPCPackExpand(pkt, sizeof(PGPInt32));
	*(PGPInt32 *)(&pkt->base[pkt->ptr]) = (PGPInt32)num;
	pkt->ptr += sizeof(PGPInt32);
}

PGPBoolean
unpack_bool(PGPPackMsg *pkt)
{
	PGPInt32 num = *(PGPInt32 *)(&pkt->base[pkt->ptr]);
	pkt->ptr += sizeof(PGPInt32);
	return num == 1 ? TRUE : FALSE;
}

void
pack_byte(PGPPackMsg *pkt, PGPByte num)
{
	PGPInt32 i = (PGPInt32)num;
	pgpRPCPackExpand(pkt, sizeof(PGPInt32));
	*(PGPInt32 *)(&pkt->base[pkt->ptr]) = i;
	pkt->ptr += sizeof(PGPInt32);
}

PGPByte
unpack_byte(PGPPackMsg *pkt)
{
	PGPInt32 num = *(PGPInt32 *)(&pkt->base[pkt->ptr]);
	pkt->ptr += sizeof(PGPInt32);
	return (PGPByte)num;
}
